#!/usr/bin/env bash

if [ $CI_COMMIT_REF_NAME != "master" ]; then
    echo "Not on master, quitting"
    exit 0
fi

set -e

SIGNING_KEY=$REPODIR/linux/repository.key

echo "Installing dependencies"
apt-get update
apt-get install -y apt-utils binutils unzip curl dpkg-sig dpkg-dev gnupg-agent expect

echo "Building directories"
mkdir repository
mkdir repository/all
mv *_all.deb repository/all/
mkdir repository/amd64
mv *_amd64.deb repository/amd64/
mkdir repository/i386
# Allow this one to fail, as there will be most likely none
mv *_i386.deb repository/i386/ || :

echo "Starting gpg-agent"
gpg --version
gpg-agent --daemon --write-env-file "${HOME}/.gpg-agent-info"
if [ -f "${HOME}/.gpg-agent-info" ]; then
    . "${HOME}/.gpg-agent-info"
    export GPG_AGENT_INFO
    export SSH_AUTH_SOCK
    export SSH_AGENT_PID
fi
GPG_TTY=$(tty)
export GPG_TTY

echo "Importing signing key"
$REPODIR/linux/import-gpg-key "${SIGNING_KEY}"

SIGNING_KEY_ID=$(gpg --list-secret-keys | grep 4096R | cut -d'/' -f2 | cut -d' ' -f1)
echo "Signing key ID: ${SIGNING_KEY_ID}"

cd repository

echo "Removing any signature"
find . -iname "*.deb" | xargs -L 1 -I '{}' ar d '{}' _gpgbuilder 2>&1 > /dev/null

echo "Signing all packages with key"
for file in */*.deb; do
    dpkg-sig --sign builder -K $SIGNING_KEY_ID $file
done

echo "Generating packages list."
apt-ftparchive packages . > Packages
gzip -c Packages > Packages.gz

echo "Generating release list."
apt-ftparchive release . > Release
gzip -c Release > Release.gz

echo "Signing release."
echo ${REPOSITORY_KEY_PASSPHRASE} | gpg --digest-algo SHA256 --batch --pinentry-mode loopback --passphrase-fd 0 --local-user "<support@transip.nl>" -o InRelease --clearsign Release 2>&1 > /dev/null
echo "Signing (2)"
echo ${REPOSITORY_KEY_PASSPHRASE} | gpg --digest-algo SHA256 --batch --pinentry-mode loopback --passphrase-fd 0 --local-user "<support@transip.nl>" -o Release.gpg -abs Release 2>&1 > /dev/null
