#!/usr/bin/env bash

if [ $CI_COMMIT_REF_NAME != "master" ]; then
    echo "Not on master, quitting"
    exit
fi

set -e

destdir=$1
signingkey=$2
signKeyID=$3
distro=$4

export REPODIR=$(pwd)

cd $destdir

echo "Installing dependencies"
yum install -y createrepo unzip rpm-sign expect

echo "Importing signing key"
$REPODIR/linux/import-gpg-key "${signingkey}"

echo "GPG: Allow loopback pinentry"
echo "allow-loopback-pinentry" > $HOME/.gnupg/gpg-agent.conf
gpg-connect-agent reloadagent /bye

echo "Exporting signing key ${signingkey} to rpm"
gpg --export -a "${signKeyID}" > "RPM-GPG-KEY"
rpm --import "RPM-GPG-KEY"

# See https://github.com/jenkinsci/rpmsign-plugin/blob/master/src/main/resources/jenkins/plugins/rpmsign/RpmSignPlugin/help-passphrase.html#L28
echo "%_signature gpg" > ~/.rpmmacros
echo "%_gpg_name ${signKeyID}" >> ~/.rpmmacros
echo "%__gpg_sign_cmd /usr/bin/gpg --no-verbose --no-armor --pinentry-mode loopback --no-secmem-warning -u \"%{_gpg_name}\" -sbo %{__signature_filename} %{__plaintext_filename}" >>  ~/.rpmmacros

echo "Signing all packages with key ${signKeyID}"
for file in */*.rpm; do
	echo "Signing: ${file}"
	# Either try it with a password entry with expect, and if that fails because the password was already entered, try it without expect:
	$REPODIR/linux/rpm-resign-cmd ${file} || rpm --resign ${file}
done

echo "Creating repository information"
createrepo .

echo "Signing release."
echo ${REPOSITORY_KEY_PASSPHRASE} | gpg --detach-sign --passphrase-fd 0 --armor --batch --local-user "${signKeyID}" repodata/repomd.xml

echo "Creating .repo file"
cat > transip-stack.repo <<EOL
[transip-stack]
name=TransIP Stack Client (${distro})
type=rpm-md
baseurl=http://mirror.transip.net/stack/software/yum/${distro}/
gpgcheck=1
gpgkey=http://mirror.transip.net/stack/release.key
enabled=1
EOL
